class User < ActiveRecord::Base
  authenticates_with_sorcery!

  validates :email, uniqueness: true
  validates :password, length: { minimum: 6 }, confirmation: true, if: :password_required?
  validates :password_confirmation, presence: true, if: :password_required?
  validates :first_name, :last_name, :nickname, :roles, presence: true

  has_many :reviews

  ROLES = %w(admin moderator)

  def add_role role
  	roles += role_mask(role) if ROLES.include?(role) && !has_role?(role)
  end

  def remove_role role
  	roles -= role_mask(role) if ROLES.include?(role) && has_role?(role)
  end

  def has_role? role
  	ROLES.include?(role) && role_mask(role) & roles == role_mask(role) 
  end

  def role_names
    names = ROLES.select{ |r| has_role? r }.join ', '
    names.size == 0 ? 'none' : names
  end

  # defines methods is_admin?, is_moderator?
  ROLES.each do |role|
  	define_method "is_#{role}?" do
      has_role?(role)
  	end
  end

  def full_name
    "#{first_name} #{last_name}"
  end

  def full_name_with_nickname
    "#{full_name} a.k.a #{nickname}"
  end

  def average_review_rating
    reviews.count == 0 ? 0 : (reviews.map(&:rating).sum.to_f / reviews.count).round(2)
  end

  def self.generous n = 5
    self.sorted_reviews -1, n
  end

  def self.tough n = 5
    self.sorted_reviews 1, n
  end

  def role_mask role
    2**ROLES.index(role)
  end

  private

    def self.sorted_reviews factor, n = 5
      User.all.reject{ |u| u.average_review_rating == 0 }.sort_by{ |u| factor * u.average_review_rating }[0..(n-1)]
    end

    def password_required?
      password.present? || password_confirmation.present? || email_changed?
    end
end
